/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef _TSCP_H_
#define _TSCP_H_

/*==================================================================================================
 Include Files
==================================================================================================*/
#include "fsl_common.h"
#include "TimersManager.h"
#include "SerialManager.h"

/*==================================================================================================
 Public macros
==================================================================================================*/

#ifndef TSCP_MAX_NUMBER_OF_INSTANCES
#define TSCP_MAX_NUMBER_OF_INSTANCES    (2U)      /*!< Maximum number of instances that can register callbacks */
#endif

#ifndef TSCP_MAX_NB_OPCODE
#define TSCP_MAX_NB_OPCODE              (15U)
#endif

#define TSCP_USE_MEM_MANAGER            1       /*!< Use MemManager from Connectivity Framework */

#define TSCP_START_CODE_PARAMS          (0x5053u)    /* SP in ASCII (Little Endian) = Params encoding (int16) */
#define TSCP_START_CODE_RAW             (0x5253u)    /* SR in ASCII (Little Endian) = Raw encoding (bytes) */

/*==================================================================================================
 Public type definitions
==================================================================================================*/
typedef void (*pTscpCallback_t) (int16_t* pParamList, uint16_t paramCount);
typedef void (*pTscpDefaultCallback_t) (uint8_t* pPacket);

typedef enum tscpOpCodes_tag{
    kTSCP_OpCodeResponse_c = 0u,
    kTSCP_OpCodeEvent_c,
    kTSCP_OpCodeStart_c,
    kTSCP_OpCodeStop_c,
    kTSCP_OpCodeSetConfiguration_c,
    kTSCP_OpCodeGetConfiguration_c,
    kTSCP_OpCodeGetData_c,
    kTSCP_OpCodeCount_c,                    /*!<This value represents the count of OpCodes and it is not an OpCode itself */
}tscpOpCodes_t;

typedef enum tscpTestCodes_tag{
    kTSCP_TestCodeUnmodulatedIq_c = 0u,
    kTSCP_TestCodeModulatedIq_c,
    kTSCP_TestPhaseDistanceEst_c
}tscpTestCodes_t;

typedef enum tscpErrorCodes_tag{
    kTSCP_ErrorOk = 0u,
    kTSCP_ErrorOpCodeNotSupported,
    kTSCP_ErrorTestNotSupported,
    kTSCP_ErrorInsuficcientParameters,
    kTSCP_ErrorExecutionFailed,
    kTSCP_ErrorUnexpectedEvent,
    kTSCP_ErrorTimeout,
    kTSCP_ErrorCount_c,                    /*!<This value represents the count of Errors and it is not an Error itself */
}tscpErrorCodes_t;

typedef enum tscpEventCodes_tag{
    kTSCP_EventTestDone_c = 0u,
    kTSCP_EventTestError_c,
    kTSCP_EventTestTimeout_c,
    kTSCP_EventCount_c,                    /*!<This value represents the count of Events and it is not an Event itself */
}tscpEventCodes_t;

typedef enum tscpSetGetConfigurationCodes_tag{
    kTSCP_SetGetConfigurationTrim_c = 0u,
    kTSCP_SetGetConfigurationCount_c      /*!<This value represents the count of Set/Get Configuration codes and is not a code itself */
}tscpSetGetConfigurationCodes_t;

typedef enum tscpTrimCommand_tag{
    kTSCP_TrimCommandStart_c = 0u,
    kTSCP_TrimCommandStop_c,
    kTSCP_TrimCommandTrimUp_c,
    kTSCP_TrimCommandTrimDown_c,
    kTSCP_TrimCommandCount_c                /*!<This value represents the count of trim commands and is not a trim command itself */
}tscpTrimCommand_t;

typedef enum tscpGetDataCodes_tag{
    kTSCP_GetDataTofDistance_c,
    kTSCP_GetDataTofSamples_c,
    kTSCP_GetDataIqSamples_c,
    kTSCP_GetDistanceEstimation_c,
    kTSCP_GetPhaseSlopeData_c,
    kTSCP_GetCdeData_c,
    kTSCP_GetDataCount_c,                 /*!<This value represents the count of Get Data Codes and it is not a Get Data Code itself */
}tscpGetDataCodes_t;

typedef struct tscpPacketStructureHeader_tag{
    uint16_t startHeader;
    uint8_t appId;
    uint8_t opCode;
    uint16_t lenght;                      /*!< either paramsNb or payloadLenght, depending on startHeader code */
}tscpPacketStructureHeader_t;

typedef struct tscpParamList_tag{
    int16_t *pParamList;
    uint16_t paramCount;
}tscpParamList_t;

#define TSCP_HEADER_OFFSET          sizeof(tscpPacketStructureHeader_t)

/*==================================================================================================
 Public functions
==================================================================================================*/

/*! ************************************************************************************************
    \brief This function packs the TSCP message in an array ready to be transmitted over the
           transport. Packing of uint16 is littleEndian.

    \param [in]  appId Instance identifying the application, as appearing in packets
    \param [in]  opCode TSCP packet OpCode
    \param [in]  pParamList Pointer to the 16-bit list of parameters
    \param [in]  pParamCount Number of parameters in the list
    \param [out] Pointer to the variable that holds the ammount of bytes in the array
    \retval     Pointer to the created buffer if success. NULL if failure
    \warning    The created array must be destroyed using TSCP_PackDestroy after being sent over
                the transport. Not doing this might result in a memory leackage.
************************************************************************************************* */
uint8_t *TSCP_PackParams (uint8_t appId, uint8_t opCode, int16_t* pParamList, uint16_t paramCount, uint16_t* pOutputBufferSize);

/*! ************************************************************************************************
    \brief This function packs the TSCP message in an array ready to be transmitted over the
           transport. Packing of uint16 is littleEndian.

    \param [in]  appId Instance identifying the application, as appearing in packets
    \param [in]  opCode TSCP packet OpCode
    \param [in]  nbListEntries  number of entries in tscpParamList_t array
    \param [in]  pParamListList Pointer to tscpParamList_t array
    \param [out] Pointer to the variable that holds the ammount of bytes in the array
    \retval     Pointer to the created buffer if success. NULL if failure
    \warning    The created array must be destroyed using TSCP_PackDestroy after being sent over
                the transport. Not doing this might result in a memory leackage.
************************************************************************************************* */
uint8_t *TSCP_PackParamsList(uint8_t appId, uint8_t opCode, uint8_t nbListEntries, tscpParamList_t *pParamListList, uint16_t* pOutputBufferSize);

/*! ************************************************************************************************
    \fn void TSCP_PackDestroy (uint8_t* pPacket)

    \brief Destroys a previously packed array using TSCP_PackTSCP_Pack

    \param [in] pPacket Pinter to the first element of the array to destroy
************************************************************************************************* */
void TSCP_PackDestroy (uint8_t* pPacket);

/*! ************************************************************************************************
    \fn void TSCP_Parse (uint8_t pPacket)

    \brief Parses a received TSCP packet and executes the registered callback for that opCode

    \param [in] instance Instance from which the registered callback will be executed
    \param [in] pPacket Pointer to the first element of the packet to parse
************************************************************************************************* */
void TSCP_Parse (uint8_t instance, uint8_t* pPacket);

/*! ************************************************************************************************
    \fn uint8_t TSCP_RegisterCallback (uint8_t opCode, pTscpCallback_t callbackFnc)

    \brief Registers a callback to be executed when the opCode is received

    \param [in] instance Instance for which the callback will be registered
    \param [in] opCode OpCode that will trigger the execution of this callback
    \param [in] callbackFnc Callback function to execute
    \retval     0 if success. 1 if failure
************************************************************************************************* */
uint8_t TSCP_RegisterCallback (uint8_t instance, uint8_t opCode, pTscpCallback_t callbackFnc);

uint8_t TSCP_RegisterDefaultCallback (uint8_t instance, pTscpDefaultCallback_t callbackFnc);

/*!
 * Initialise TSCP over UART.
 * \param [in] appSerId Application serialId as passed to Serial_InitInterface
 * \param [in] maxWaitTimeInSec Time used to detect incomplete message sent over UART. If time expires, incomplete message will be discarded.
 */
void TSCP_UartInit(uint8_t appSerId, tmrTimeInSeconds_t maxWaitTimeInSec);

/*!
 * Bind an appId to an instance, in order to receive only packets targetted to this appId.
 * \param [in] appId Application identifier as defined in the packet header
 * \param [in] instance TSCP instance (callbacks to be invoked)
 */
serialStatus_t TSCP_UartBind(uint8_t appId, uint8_t instance);

/*!
 * Format and send on UART, a message built from a paramlist and opCode.
 * \param [in]  appId identifying the application, as appearing in packets
 * \param [in] opCode 
 * \param [in] pParamList List of parameters (as an array of int16_t)
 * \param [in] paramCount
 */
serialStatus_t TSCP_UartSyncWriteParams(uint8_t appId, uint8_t opCode, int16_t *pParamList, uint16_t paramCount);

/*!
 * Format and send on UART, a message built from a paramlist and opCode.
 * \param [in]  appId identifying the application, as appearing in packets
 * \param [in] opCode 
 * \param [in]  nbListEntries  number of entries in tscpParamList_t array
 * \param [in]  pParamListList Pointer to tscpParamList_t array
 */
serialStatus_t TSCP_UartSyncWriteParamsList(uint8_t instance, uint8_t opCode, uint8_t nbListEntries, tscpParamList_t *pParamListList);

/*!
 * Format and send on UART, a message built from a buffer pointer and opCode.
 * \param [in]  appId identifying the application, as appearing in packets
 * \param [in] opCode 
 * \param [in] pBuffer byte buffer
 * \param [in] bufLen Lenght of buffer in bytes
 */
serialStatus_t TSCP_UartSyncWriteRaw(uint8_t appId, uint8_t opCode, uint8_t *pbuffer, uint16_t bufLen);

#endif //_TSCP_H_
